package com.edgeti.EdgeUtils.layout.managers
{
	import com.edgeti.EdgeUtils.DynamicPanel.DynamicPanel;
	import com.edgeti.EdgeUtils.DynamicPanel.PanelEvent;
	
	import flash.display.DisplayObject;
	
	import mx.collections.ArrayCollection;
	import mx.containers.Canvas;
	import mx.core.UIComponent;
	import mx.effects.Move;
	import mx.effects.Parallel;
	import mx.effects.Resize;
	import mx.effects.Sequence;
	import mx.events.EffectEvent;
	import mx.events.ResizeEvent;

	/**
	 * A very basic LayoutManager which allows the free dragging/resizing
	 * of panels and when refreshed, will cascard the panels from top
	 * left to bottom right.
	 * 
	 * Minimized items are moved to the bottom left of the screen and fill
	 * horizontally across until a new row is needed.
	 * 
	 * **/
	public class CascadeLayoutManager extends LayoutManagerBase
	{
		protected static const VERTICAL_OFFSET:int = 25;
		protected static const HORIZONTAL_OFFSET:int = 25;
		protected static const BUFFER:int = 5;
		
		protected var _minimizePanels:ArrayCollection;
		protected var _miniParallel:Parallel;
				
		public function CascadeLayoutManager(view:Canvas)
		{
			super(view);
			
			name = "Cascade";
			_minimizePanels = new ArrayCollection();
		}
		
		// Returns whether or not the main transition is occurring.
		override public function get transitioning():Boolean
		{
			var _isMiniPlaying:Boolean = _miniParallel == null ? false : _miniParallel.isPlaying;
			
			return _isMiniPlaying || super.transitioning;
		}
		
		override public function stopTransition():void
		{
			super.stopTransition();
			if (_miniParallel != null){
				_miniParallel.pause();
			}
		}
		
		//************************************
		// Apply / Reset Layout
		//************************************
		
		override public function applyLayout(animate:Boolean):void{
			var panel:DynamicPanel;
			var i:int;
			
			var minimizedPanels:Array = new Array();
			var maximizedPanels:Array = new Array();
			var defaultPanels:Array = new Array();
			
			//Split panels by their state
			for (i = 0; i < items.length; ++i){
				panel = items.getItemAt(i) as DynamicPanel;
				switch (panel.windowState){
					case DynamicPanel.WINDOW_STATE_DEFAULT:
						defaultPanels.push(panel);
						break;
					case DynamicPanel.WINDOW_STATE_MAXIMIZED:
						maximizedPanels.push(panel);
						break;
					case DynamicPanel.WINDOW_STATE_MINIMIZED:
						minimizedPanels.push(panel);
						break;
				}
			}
			
			//This saves a lot of duplicate code
			var duration:Number = animate ? 1000 : 0;
			
			applyLayoutHelper(defaultPanels, maximizedPanels, minimizedPanels, duration);
		}
		
		protected function applyLayoutHelper(defaultPanels:Array, maxiPanels:Array, 
												miniPanels:Array, duration:Number):void{
			var panel:DynamicPanel;
			var xTo:Number;
			var yTo:Number;
			var wTo:Number;
			var hTo:Number;
			var i:int;
			var move:Move;
			var resize:Resize;
			
			var animations:Array = new Array();
			//Panels in default state
			for (i = 0; i < defaultPanels.length; ++i){
				panel = defaultPanels[i] as DynamicPanel;
				xTo = HORIZONTAL_OFFSET*i;
				yTo = VERTICAL_OFFSET*i;
				move = createMoveEffectInstance(panel, xTo, yTo);
				//move == null if no move is required
				if (move){
					animations.push(move);
				}
			}
			//Panels in maximized state
			for (i = 0; i < maxiPanels.length; ++i){
				panel = maxiPanels[i] as DynamicPanel;
				xTo = BUFFER;
				yTo = BUFFER;
				move = createMoveEffectInstance(panel, xTo, yTo);
				//move == null if no move is required
				if (move){
					animations.push(move);
				}
				wTo = getViewWidth() - 2 * BUFFER;
				hTo = getViewHeight() - 2 * BUFFER;
				resize = createResizeEffectInstance(panel, wTo, hTo);
				//resize == null if no resize is required
				if (resize){
					animations.push(resize);
				}
			}
			//Panels in minimized state
			for (i = 0; i < miniPanels.length; ++i){
				_minimizePanels = new ArrayCollection(miniPanels);
				positionMinimized();
			}
			
			// Only animate if there are items to animate.
			if (animations.length > 0)
			{
				_parallel = new Parallel();
				_parallel.children = animations;
				_parallel.duration = duration;
				_parallel.play();
				_parallel.addEventListener(EffectEvent.EFFECT_END, onEffectEnd);
			}
		}
		
		//************************************
		// Managed Items Changes
		//************************************
		
		override protected function addPanelListeners(panel:DynamicPanel):void{
			super.addPanelListeners(panel);
			
			//panel.allowClose = true;
			panel.allowResize = panel.windowState == DynamicPanel.WINDOW_STATE_DEFAULT;
			panel.allowMinimize = true;
			panel.allowMaximize = true;
			panel.allowMove = panel.windowState == DynamicPanel.WINDOW_STATE_DEFAULT;
		}
		
		override public function set items(val:ArrayCollection):void{
			_minimizePanels = new ArrayCollection();
			super.items = val;
		}
		
		override protected function onItemAdd(items:Array, location:int):void{
			super.onItemAdd(items, location);
			var xTo:Number;
			var yTo:Number;
			var wTo:Number;
			var hTo:Number;
			
			//Pretty safe assumption only one panel added at atime
			var panel:DynamicPanel = items[0] as DynamicPanel;
			switch (panel.windowState){
				case DynamicPanel.WINDOW_STATE_DEFAULT:
					//By default, add new panels to top left
					xTo = BUFFER;
					yTo = BUFFER;
					//If we can, add the new child slightly offset from the last
					if (_view.numChildren > 0){
						var previousChild:DisplayObject = _view.getChildAt(_view.numChildren - 1);
						if (previousChild is DynamicPanel && 
							(previousChild as DynamicPanel).windowState == DynamicPanel.WINDOW_STATE_DEFAULT){
							xTo = previousChild.x + HORIZONTAL_OFFSET;
							yTo = previousChild.y + VERTICAL_OFFSET;
						}
						
					}
					panel.move(xTo, yTo);
					break;
				case DynamicPanel.WINDOW_STATE_MAXIMIZED:
					xTo = BUFFER;
					yTo = BUFFER;
					wTo = getViewWidth() - 2 * BUFFER;
					hTo = getViewHeight() - 2 * BUFFER;
					
					panel.move(xTo, yTo);
					panel.width = wTo;
					panel.height = hTo;
					break;
				case DynamicPanel.WINDOW_STATE_MINIMIZED:
					if (!_minimizePanels.contains(panel)){
						_minimizePanels.addItem(panel);
					}
					positionMinimized();
					break;
			}
		}
		
		override protected function onItemRemove(items:Array, location:int):void{
			var change:Boolean = false;
			for each (var panel:DynamicPanel in items){
				if (_minimizePanels.contains(panel)){
					_minimizePanels.removeItemAt(_minimizePanels.getItemIndex(panel));
					change = true;
				}
			}
			if (change){
				positionMinimized();
			}
			
			super.onItemRemove(items, location);
			//Don't care for this layout since it must be manually refreshed
		}
		
		override protected function onItemReset():void{
			super.onItemReset();
			applyLayout(false);
		}
		
		//************************************
		// Minimize / Maximize
		//************************************
		
		override protected function onPanelMinimize(evt:PanelEvent):void{
			var panel:DynamicPanel = evt.target as DynamicPanel;
			if (!_minimizePanels.contains(panel)){
				_minimizePanels.addItem(panel);
			}
			panel.allowMove = false;
			positionMinimized();
		}
		
		override protected function onPanelMaximize(evt:PanelEvent):void{
			var panel:DynamicPanel = evt.target as DynamicPanel;
			//Move
			var xTo:Number = BUFFER;
			var yTo:Number = BUFFER;
			//Resize
			var wTo:Number = getViewWidth() - 2 * BUFFER;
			var hTo:Number = getViewHeight() - 2 * BUFFER;
			
			movePanel(panel, xTo, yTo, wTo, hTo, true);
		}
		
		override protected function onPanelRestore(evt:PanelEvent):void{
			var panel:DynamicPanel = evt.target as DynamicPanel;
			if (_minimizePanels.contains(panel)){
				_minimizePanels.removeItemAt(_minimizePanels.getItemIndex(panel));
				positionMinimized();
			}
			panel.allowResize = true;
			panel.allowMove = true;
			
			var moveFirst:Boolean = evt.previousState == DynamicPanel.WINDOW_STATE_MINIMIZED;
			
			movePanel(panel, evt.oldX, evt.oldY, evt.oldWidth, evt.oldHeight, moveFirst);
		}
		
		protected function movePanel(panel:UIComponent, xTo:Number, yTo:Number,
										wTo:Number, hTo:Number, moveFirst:Boolean):void{
			
			var move:Move = createMoveEffectInstance(panel, xTo, yTo);
			var resize:Resize = createResizeEffectInstance(panel, wTo, hTo);
			
			var animations:Array = new Array();
			if (resize){
				animations.push(resize);
			}
			if (move){
				animations.push(move);
			}
			
			if (moveFirst){
				animations.reverse();
			}
			
			var sequence:Sequence = new Sequence();
			sequence.children = animations;
			sequence.duration = 400;
			sequence.play(); 
		}
		
		protected function positionMinimized():void{
			if (_minimizePanels.length == 0){
				return;
			}
			
			var containerWidth:Number = getViewWidth();
			var minimizedWidth:Number = DynamicPanel.MIN_WIDTH;
			var minimizedHeight:Number = DynamicPanel.MIN_HEIGHT;
			var levels:int = 1;
			var panelsPerLevel:int = 99;
			var verticalOffset:Number = 1;
			var totalWidth:Number = (minimizedWidth + BUFFER) * _minimizePanels.length;
			
			if (totalWidth > containerWidth){
				panelsPerLevel = int(containerWidth/(minimizedWidth + BUFFER));
				panelsPerLevel = panelsPerLevel > 0 ? panelsPerLevel : 1;
				levels += int (_minimizePanels.length/panelsPerLevel);
			}
			
			var panel:DynamicPanel;
			var level:int;
			var panelIndex:int;
			var xTo:Number;
			var yTo:Number;
			for (var l:int = 1; l <= levels; ++l){
				for (var i:int = 0; ; ++i){
					panelIndex = (l-1)*panelsPerLevel + i;
					if (i > panelsPerLevel - 1 || panelIndex >= _minimizePanels.length){
						break;
					}
					panel = _minimizePanels.getItemAt(panelIndex) as DynamicPanel;
					xTo = i * (BUFFER + minimizedWidth);
					yTo = getViewHeight() - ((minimizedHeight + 3) * l) - verticalOffset;
					movePanel(panel, xTo, yTo, minimizedWidth, minimizedHeight, false);
				}
			}
		}
		
		//************************************
		// Container Resize
		//************************************
		
		override protected function onContainerResize(evt:ResizeEvent):void{
			super.onContainerResize(evt);
			
			positionMinimized();
			
			for each (var panel:DynamicPanel in _items){
				if (panel.windowState == DynamicPanel.WINDOW_STATE_MAXIMIZED){
					panel.width = getViewWidth() - 2 * BUFFER;
					panel.height = getViewHeight() - 2 * BUFFER;
				}
			}
			
		}
		
		//************************************
		// Focus Item
		//************************************
		
		override public function focusPanel(panel:UIComponent):void{
			_view.setChildIndex(panel, _view.numChildren - 1);
		}
	}
}