package com.edgeti.EdgeUtils.DynamicPanel
{
	import com.edgeti.EdgeUtils.DynamicPanel.skins.candy.*;
	import com.edgeti.EdgeUtils.logger.Logger;
	
	import flash.events.Event;
	import flash.events.FocusEvent;
	import flash.events.MouseEvent;
	import flash.geom.Rectangle;
	
	import mx.containers.HBox;
	import mx.containers.Panel;
	import mx.controls.Button;
	import mx.controls.TextInput;
	import mx.core.Application;
	import mx.core.EdgeMetrics;
	import mx.core.FlexVersion;
	import mx.effects.Move;
	import mx.effects.Resize;
	import mx.events.FlexEvent;

	[Event(name="close", type="com.edgeti.EdgeUtils.DynamicPanel.PanelEvent")]
	[Event(name="minimize", type="com.edgeti.EdgeUtils.DynamicPanel.PanelEvent")]
	[Event(name="maximize", type="com.edgeti.EdgeUtils.DynamicPanel.PanelEvent")]
	[Event(name="default", type="com.edgeti.EdgeUtils.DynamicPanel.PanelEvent")]
	[Event(name="resizeEnd", type="com.edgeti.EdgeUtils.DynamicPanel.PanelEvent")]

	/**
	 * The legendary DynamicPanel.  DynamicPanels may be resized, minimized, 
	 * maximized, closed, retitled and moved by dragging all at run-time. All
	 * of these features can be enabled/disabled with the following properties:
	 * 
	 * allowClose
	 * allowMove
	 * allowResize
	 * allowMinimize
	 * allowMaximize
	 * allowRename
	 * 
	 * The current state of the panel (minimized, maximized, or default) is stored
	 * in the windowState property.  To programmatically change the state either call:
	 * 
	 * windowState = DynamicPanel.WINDOW_STATE_MINIMIZED //for any of the three states
	 * 
	 * or there are the following methods:
	 * 
	 * minimizePanel()
	 * maximizePanel()
	 * 
	 * and there's always closePanel().
	 * 
	 * DynamicPanels are designed to work with LayoutManagers.  If a panel is added to
	 * a ManagedCanvas, it will no longer resize itself when the minize or maximize buttons
	 * are clicked.  Instead it leaves the sizing and positioning up to the manager.
	 * 
	 * If a panel is not managed, minimize will make the window short and 200 pixels wide
	 * while maximize will fill the parent container.
	 * 
	 * **/
	public class DynamicPanel extends Panel
	{
		//Logger
		protected var logger:Logger = Logger.getLogger();
		
		//States
		protected var _windowState:String = WINDOW_STATE_DEFAULT;
		public static const WINDOW_STATE_DEFAULT:String = "DynamicPanel.DEFAULT";
		public static const WINDOW_STATE_MINIMIZED:String = "DynamicPanel.MINIMIZED";
		public static const WINDOW_STATE_MAXIMIZED:String = "DynamicPanel.MAXIMIZED";
		
		//Permissions
		protected var _allowClose:Boolean = true;
		protected var _allowMove:Boolean = true;
		protected var _allowResize:Boolean = true;
		public var allowMinimize:Boolean = true;
		public var allowMaximize:Boolean = true;
		public var allowRename:Boolean = true;
		public var allowSearch:Boolean = false;
		
		//Buttons
		protected var _buttonContainer:HBox;
		protected var _closeButton:Button;
		protected var _minimizeButton:Button;
		protected var _maximizeButton:Button;
		protected var _resizeButton:Button;
		protected var _searchButton:Button;
		
		protected var _searchText:TextInput;
		protected var _maxSetHeight:Number;
		
		public var isManaged:Boolean = false;
		protected var _creationComplete:Boolean = false;
		
		public function DynamicPanel()
		{
			super();
		}
		
		/**
		 * Called upon creationComplete so some properties that were
		 * set may be applied such as creating the window in a minimized
		 * state.
		 * 
		 * **/
		protected function onCreationComplete(evt:FlexEvent):void{
			_creationComplete = true;
			invalidateProperties();
		}
		
		//************************************
		// Layout
		//************************************
		
		/**
		 * This method adds all the allowed buttons and event listeners. 
		 * **/
		override protected function createChildren():void{
			super.createChildren();
			addEventListener(FlexEvent.CREATION_COMPLETE, onCreationComplete);
			
			_buttonContainer = new HBox();
			_buttonContainer.width = 2;
			_buttonContainer.height = 15;
			_buttonContainer.setStyle("paddingLeft", 0);
			_buttonContainer.setStyle("paddingRight", 0);
			_buttonContainer.setStyle("horizontalGap", 5);
			
			
			
			//Resize button always exists, visiblity changes
			_resizeButton = new Button();
			_resizeButton.visible = _allowResize;
			_resizeButton.toolTip = "Resize";
			_resizeButton.width = 10;
			_resizeButton.height = 10;
			_resizeButton.setStyle("skin", resizeButton);
			_resizeButton.addEventListener(MouseEvent.MOUSE_DOWN, resizeDownHandler);
			rawChildren.addChild(_resizeButton);
			
			//Allow move event listeners always on
			titleBar.addEventListener(MouseEvent.MOUSE_DOWN, startDragHandler);
			titleBar.addEventListener(MouseEvent.MOUSE_UP, stopDragHandler);
			
			titleBar.doubleClickEnabled = true;
			titleBar.addEventListener(MouseEvent.DOUBLE_CLICK, renameHandler);
			titleBar.addEventListener(MouseEvent.MOUSE_OVER, hoverHandler);
			
			
			if(allowSearch){
				_searchButton = new Button();
				_searchButton.label = "Search";
				_searchButton.width = 70;
				_searchButton.height = 15;
				_searchButton.addEventListener(MouseEvent.CLICK, doSearch);
				_buttonContainer.addChild(_searchButton);
				_buttonContainer.width += 40;
				_searchText = new TextInput();
				_searchText.width = 80;
				_buttonContainer.addChild(_searchText);
				_buttonContainer.width += 120;
			}
			if (allowMinimize){
				_minimizeButton = new Button();
				initButton(_minimizeButton, "Minimize", minimizeButton);
				_minimizeButton.addEventListener(MouseEvent.CLICK, minimizeHandler);
				
				_buttonContainer.width += 20;
				_buttonContainer.addChild(_minimizeButton);
			}
			if (allowMaximize){
				_maximizeButton = new Button();
				initButton(_maximizeButton, "Maximize", maximizeButton);
				_maximizeButton.addEventListener(MouseEvent.CLICK, maximizeHandler);
				
				_buttonContainer.width += 20;
				_buttonContainer.addChild(_maximizeButton);
			}
			if (allowClose){
				_closeButton = new Button();
				initButton(_closeButton, "Close", closeButton);
				_closeButton.addEventListener(MouseEvent.CLICK, closeHandler);
				
				_buttonContainer.width += 20;
				_buttonContainer.addChild(_closeButton);
			}
			
			titleBar.addChild(_buttonContainer);
			addEventListener(MouseEvent.CLICK, mouseClickHandler);
		}
		
		/**
		 * Reused method for initializing buttons.
		 * **/
		protected function initButton(button:Button, tip:String, skin:Class):void{
			button.toolTip = tip;
			button.setStyle("skin", skin);
			button.width = 50;
			button.height = 50;
			button.scaleX = 0.3;
			button.scaleY = 0.3;
		}
		
		/**
		 * Positions the button box in the titleBar as well as the resize button in the lower right.
		 * **/
		override protected function updateDisplayList(uw:Number, uh:Number):void{
			super.updateDisplayList(uw, uh);
			
			_buttonContainer.x = uw - 5 - _buttonContainer.width;
			_buttonContainer.y = 8;
			
			if (_resizeButton){
				_resizeButton.x = uw - _resizeButton.width;
				_resizeButton.y = uh - _resizeButton.height;
			}
			
			var bm:EdgeMetrics = FlexVersion.compatibilityVersion < FlexVersion.VERSION_3_0 ? 
									borderMetrics :
									EdgeMetrics.EMPTY;
			var reqWidth:Number = bm.left + bm.right;
			
			var offset:Number = 10;
			reqWidth += offset * 2;
			reqWidth += _buttonContainer.width;
			
			titleTextField.width = uw - reqWidth;
		}
		
		/**
		 * Attemtps to clear all event listenres before removing the panel.
		 * **/
		public function dispose():void{
			removeEventListener(MouseEvent.CLICK, mouseClickHandler);
			
			//Drag Handlers
			titleBar.removeEventListener(MouseEvent.MOUSE_DOWN, startDragHandler);
			titleBar.removeEventListener(MouseEvent.MOUSE_UP, stopDragHandler);
			
			//Button Handlers
			if (_closeButton)
				_closeButton.removeEventListener(MouseEvent.CLICK, closeHandler);
			if (_maximizeButton)
				_maximizeButton.removeEventListener(MouseEvent.CLICK, maximizeHandler);
			if (_minimizeButton)
				_minimizeButton.removeEventListener(MouseEvent.CLICK, minimizeHandler);
			if (_resizeButton)
				_resizeButton.removeEventListener(MouseEvent.MOUSE_DOWN, resizeDownHandler);
			
		}
		
		//************************************
		// Permissions
		//************************************
		[Bindable]
		public function set allowClose(val:Boolean):void{
			_allowClose = val;
		}
		
		public function get allowClose():Boolean{
			return _allowClose;
		}
		
		[Bindable]
		public function set allowResize(val:Boolean):void{
			_allowResize = val;
			if (_resizeButton){
				_resizeButton.visible = _allowResize;
			}
		}
		
		public function get allowResize():Boolean{
			return _allowResize;
		}
		
		[Bindable]
		public function set allowMove(val:Boolean):void{
			_allowMove = val;
		}
		
		public function get allowMove():Boolean{
			return _allowMove;
		}
		
		//************************************
		// Interface
		//************************************
		
		/**
		 * Closes the panel.  Closed panels may not be recovered.
		 * **/
		public function closePanel():void{
			closeHandler();
		}
		
		/**
		 * If the panel is in the minimized state, this will 
		 * restore it a default state, otherwise it will be 
		 * minimized.
		 * **/
		public function minimizePanel():void{
			if (windowState == WINDOW_STATE_MINIMIZED){
				windowState = WINDOW_STATE_DEFAULT;
			}
			else {
				windowState = WINDOW_STATE_MINIMIZED;
			}
		}
		
		/**
		 * If the panel is in the maximized state, this will 
		 * restore it a default state, otherwise it will be 
		 * maximized.
		 * **/
		public function maximizePanel():void{
			if (windowState == WINDOW_STATE_MAXIMIZED){
				windowState = WINDOW_STATE_DEFAULT;
			}
			else {
				windowState = WINDOW_STATE_MAXIMIZED;
			}
		}
		
		/**
		 * force the panel to MINIMIZED
		 * **/
		public function forceMinimizePanel():void{
			windowState = WINDOW_STATE_MINIMIZED;
		}
		
		/**
		 * force the panel to MAXIMIZED
		 * **/
		public function forceMaximizePanel():void{
			windowState = WINDOW_STATE_MAXIMIZED;
		}
		
		/**
		 * force the panel to WINDOW_STATE_DEFAULT
		 * **/
		public function forceDefaultPanel():void{
			windowState = WINDOW_STATE_DEFAULT;
		}
		
		/**
		 * force the panel to MINIMIZED
		 * **/
		public function forceMinimizePanel2():void{
			minimize();
		}
		
		/**
		 * force the panel to MAXIMIZED
		 * **/
		public function forceMaximizePanel2():void{
			maximize();
		}
		
		/**
		 * force the panel to WINDOW_STATE_DEFAULT
		 * **/
		public function forceDefaultPanel2():void{
			restore()
		}
		
		//************************************
		// Properties
		//************************************
		
		protected var _windowStateDirty:Boolean = false;
		protected var _previousState:String;
		
		/**
		 * Changes the state of the window.  When windowState is set, 
		 * the state is not immediately changed.  Instead the request is stored
		 * until it can be changed (for example windowState may be set to minimized
		 * before the panel is even added as a child to something so it is created
		 * in a minimized state).
		 * **/
		public function set windowState(state:String):void{
			//If we're currently in default state, save measurements
			//before making any changes
			if (_windowState == WINDOW_STATE_DEFAULT){
				saveOldMeasurements();
			}
			
			if (state == WINDOW_STATE_DEFAULT &&
				windowState != WINDOW_STATE_DEFAULT){
					_previousState = windowState;
					_windowStateDirty = true;
					_windowState = WINDOW_STATE_DEFAULT;
					invalidateProperties();
			} else if (state == WINDOW_STATE_MAXIMIZED &&
				windowState != WINDOW_STATE_MAXIMIZED){
					_previousState = windowState;
					_windowStateDirty = true;
					_windowState = WINDOW_STATE_MAXIMIZED;
					invalidateProperties();
			} else if (state == WINDOW_STATE_MINIMIZED &&
				windowState != WINDOW_STATE_MINIMIZED){
					_previousState = windowState;
					_windowStateDirty = true;
					_windowState = WINDOW_STATE_MINIMIZED;
					invalidateProperties();
			}
		}
		
		public function get windowState():String{
			return _windowState;
		}
		
		/**
		 * Windowstate changes are actually handled here.
		 * **/
		override protected function commitProperties():void{
			super.commitProperties();
			
			if (_windowStateDirty && _creationComplete){
				_windowStateDirty = false;
				//Maximize
				if (windowState == WINDOW_STATE_MAXIMIZED){
					
					if (_resizeButton){
						_resizeButton.visible = false;
					}
					if (allowSearch){
						if (_searchButton){
							_searchButton.visible = true;
						}
						if (_searchText){
							_searchText.visible = true;
						}
					}
					
					dispatchEvent(new PanelEvent(PanelEvent.MAXIMIZE));
					if (!isManaged){
						maximize();
					} 
				//Minimize
				} else if (windowState == WINDOW_STATE_MINIMIZED){
					
					if (_resizeButton){
						_resizeButton.visible = false;
					}
					if (_searchButton){
						_searchButton.visible = false;
					}
					if (_searchText){
						_searchText.visible = false;	
					}
					
					dispatchEvent(new PanelEvent(PanelEvent.MINIMIZE));
					if (!isManaged){
						minimize();
					} 
				//Default
				} else if (windowState == WINDOW_STATE_DEFAULT){
					var evt:PanelEvent = new PanelEvent(PanelEvent.DEFAULT);
					evt.oldX = _oldX;
					evt.oldY = _oldY;
					evt.oldWidth = _oldW;
					evt.oldHeight = _oldH;
					evt.previousState = _previousState;
					
					if (_allowResize && _resizeButton){
						_resizeButton.visible = true;
					}
					
					if (allowSearch){
						if (_searchButton){
							_searchButton.visible = true;
						}
						if (_searchText){
							_searchText.visible = true;
						}
					}
					
					dispatchEvent(evt);
					if (!isManaged){
						restore();
					} 
				}
			}
		}
		
		//************************************
		// Event Handlers
		//************************************
		
		protected function mouseClickHandler(evt:MouseEvent):void{
			if (allowMove && this.parent){
				//Move this panel to the top of the z order
				this.parent.setChildIndex(this, this.parent.numChildren - 1);
			}
		}
		
		protected function minimizeHandler(evt:MouseEvent=null):void{
			minimizePanel();
		}
		
		protected function maximizeHandler(evt:MouseEvent=null):void{
			maximizePanel();
		}
		
		protected function closeHandler(evt:MouseEvent=null):void{
			if (isManaged){
				dispatchEvent(new PanelEvent(PanelEvent.CLOSE));
			} else {
				dispose();
				parent.removeChild(this);
			}
		}
		
		// Dragging
		protected function startDragHandler(evt:MouseEvent):void{
			if (allowMove){
				this.parent.setChildIndex(this, this.parent.numChildren - 1);
				addEventListener(MouseEvent.MOUSE_MOVE, mouseMoveHandler);
				startDrag();
			}
		}
		
		protected function mouseMoveHandler(evt:MouseEvent):void{
			//If Flex misses a beat and the mouse moves out of the titleBar,
			//the panel can "stick" to the mouse. This prevnts that
			//from happening by checking that the mouse is still
			//over the titleBar
			var rect:Rectangle = new Rectangle(titleBar.x, titleBar.y, titleBar.width, titleBar.height);
			if (!rect.contains(evt.localX, evt.localY)){
				stopDragHandler();
			}
			
			evt.updateAfterEvent();
		}
		
		protected function stopDragHandler(evt:MouseEvent=null):void{
			removeEventListener(MouseEvent.MOUSE_MOVE, mouseMoveHandler);
			stopDrag();
			
			//Don't let the panel be dropped out of reach
			if (x < 0){
				x = 0;
			}
			if (y < 0){
				y = 0;
			}
		}
		
		protected function hoverHandler(evt:MouseEvent):void{
			titleBar.toolTip = title;
		}
		
		//Renaming
		protected var _titleInput:TextInput;
		
		protected function renameHandler(evt:MouseEvent):void{
			if (allowRename && titleTextField.visible){
				if (!_titleInput){
					_titleInput = new TextInput();
					_titleInput.x = titleTextField.x;
					_titleInput.y = titleTextField.y;
					_titleInput.addEventListener(FocusEvent.FOCUS_OUT, renameDoneHandler);
					_titleInput.addEventListener(FlexEvent.ENTER, renameDoneHandler);
					titleBar.addChild(_titleInput);
				}
				_titleInput.width = titleTextField.width;
				_titleInput.height = titleTextField.height;
				_titleInput.text = title;
			}
		}
		
		protected function renameDoneHandler(evt:Event):void{
			if (_titleInput){
				_titleInput.removeEventListener(FocusEvent.FOCUS_OUT, renameDoneHandler);
				_titleInput.removeEventListener(FlexEvent.ENTER, renameDoneHandler);
				title = _titleInput.text;
				_titleInput.visible = false;
				titleBar.removeChild(_titleInput);
				_titleInput = null;
			}
		}
		
		//Resizing
		protected function resizeDownHandler(evt:MouseEvent):void{
			Application.application.parent.addEventListener(MouseEvent.MOUSE_MOVE, mouseResizeHandler);
			Application.application.parent.addEventListener(MouseEvent.MOUSE_UP, resizeUpHandler);
			_resizeButton.startDrag();
		}
		
		protected function mouseResizeHandler(evt:MouseEvent):void{
			var w:Number = _resizeButton.x + _resizeButton.width;
			var h:Number = _resizeButton.y + _resizeButton.height;
			
			if (w < MIN_WIDTH){
				w = MIN_WIDTH;
			}
			if (h < MIN_HEIGHT){
				h = MIN_HEIGHT;
			}
			
			width = w;
			height = h;
			evt.updateAfterEvent();
		}
		
		protected function resizeUpHandler(evt:MouseEvent):void{
			_resizeButton.stopDrag();
			
			Application.application.parent.removeEventListener(MouseEvent.MOUSE_MOVE, mouseResizeHandler);
			Application.application.parent.removeEventListener(MouseEvent.MOUSE_UP, resizeUpHandler);
			
			_resizeButton.x = unscaledWidth - _resizeButton.width;
			_resizeButton.y = unscaledHeight - _resizeButton.height;
			
			dispatchEvent(new PanelEvent(PanelEvent.RESIZE_END));
		}
		
		protected function doSearch(evt:Event):void{
			logger.debug("searching for: '"+_searchText.text+"'");
		}
		
		//************************************
		// Min / Max / Restore
		//************************************
		
		protected var _oldW:Number;
		protected var _oldH:Number;
		protected var _oldX:Number;
		protected var _oldY:Number;
		
		protected static const BUFFER:int = 5;
		public static const MIN_HEIGHT:int = 30;
		public static const MIN_WIDTH:int = 200;
		
		protected function minimize():void{
			//Minimize window size
			resizeTo(MIN_WIDTH, MIN_HEIGHT);
		}
		
		protected function maximize():void{
			//Maximize window size
			var newW:Number = parent.width - 2 * BUFFER;
			var newH:Number = parent.height - 2 * BUFFER;
			var newX:Number = BUFFER;
			var newY:Number = BUFFER;
			
			moveTo(newX, newY);
			resizeTo(newW, newH);
		}
		
		protected function restore():void{
			moveTo(_oldX, _oldY);
			resizeTo(_oldW, _oldH);
		}
		
		protected function saveOldMeasurements():void{
			_oldW = width;
			_oldH = height;
			_oldX = x;
			_oldY = y;
		}
		
		//Effects
		protected function resizeTo(toW:Number, toH:Number):void{
			var resize:Resize = new Resize(this);
			resize.heightTo = toH;
			resize.widthTo = toW;
			resize.duration = 400;
			resize.play();
		}
		
		protected function moveTo(toX:Number, toY:Number):void{
			var move:Move = new Move(this);
			move.xTo = toX;
			move.yTo = toY;
			move.duration = 400;
			move.play();
		}
		
		public function set maxSetHeight (h:Number) : void {
			_maxSetHeight = (h > this.height ? h : this.height);
		}
		
		
		public function get maxSetHeight () : Number {
			return _maxSetHeight;
		}
	}
}