package com.edgeti.EdgeUtils.logger
{
	import mx.collections.ArrayCollection;
	
	/**
	 * Central class of the Flex logger.  Most logging operations
	 * are done through this class.
	 * **/
	public class Logger
	{
		/**
		 * Maximum number of entries retained in the queue.
		 * **/
		public static const maxEntries:Number = 500;
		/**
		 * Whether or not to parse location information
		 * (class, method, line, file)
		 * **/
		public static var extendedLogging:Boolean = true;
		
		//Log Entries
		[Bindable]
		/**
		 * Current queue of LoggingEntries.
		 * **/
		public static var entries:ArrayCollection = new ArrayCollection();
		
		//********************************************
		// Levels
		//********************************************
		
		public static const ALL:int = LoggingEntry.ALL;
		public static const DEBUG:int = LoggingEntry.DEBUG;
		public static const INFO:int = LoggingEntry.INFO;
		public static const WARN:int = LoggingEntry.WARN;
		public static const ERROR:int = LoggingEntry.ERROR;
		public static const FATAL:int = LoggingEntry.FATAL;
		
		//********************************************
		// Singleton
		//********************************************
		
		protected static var instance:Logger;
		
		/**
		 * Do not call contructor directly, instead use Logger.getLogger().
		 * **/
		public function Logger()
		{
		}
		
		/**
		 * Provides access to the Logger instance.  At the top of every class
		 * which requires logging, use:
		 * 
		 * protected var logger:Logger = Logger.getInstance();
		 * 
		 * **/
		public static function getLogger():Logger{
			if (!instance){
				instance = new Logger();
			}
			
			return instance;
		}
		
		//********************************************
		// Log Methods
		//********************************************
		
		/**
		 * Log a message under any level.  message may be of type String,
		 * Error, or Object.  Strings are used directly.  Error's messages
		 * and stackTraces are logged.  Objects are logged using the toString()
		 * method.
		 * **/
		public function log(level:int, message:Object):void{
			var entry:LoggingEntry = new LoggingEntry();
			
			entry.level = level;
			entry.timestamp = new Date().time;
			
			if (message is String){			//String
				entry.message = message as String;
			} else if (message is Error){	//Error
				var err:Error = message as Error;
				entry.message = err.message;
				entry.throwableString = err.getStackTrace().split("\n");
			} else {						//Everything else
				entry.message = message.toString();
			}
			
			if (extendedLogging){
				parseLocation(entry);
			}
			
			if (entries.length >= maxEntries){
				entries.removeItemAt(entries.length - 1);
			}
			
			trace(entry.toString());
			entries.addItemAt(entry, 0);
			
		}
		
		/**
		 * Log a message under the level of DEBUG.  message may be of type String,
		 * Error, or Object.  Strings are used directly.  Error's messages
		 * and stackTraces are logged.  Objects are logged using the toString()
		 * method.
		 * **/
		public function debug(message:Object):void{
			log(DEBUG, message);
		}
		
		/**
		 * Log a message under the level of INFO.  message may be of type String,
		 * Error, or Object.  Strings are used directly.  Error's messages
		 * and stackTraces are logged.  Objects are logged using the toString()
		 * method.
		 * **/
		public function info(message:Object):void{
			log(INFO, message);
		}
		
		/**
		 * Log a message under the level of WARN.  message may be of type String,
		 * Error, or Object.  Strings are used directly.  Error's messages
		 * and stackTraces are logged.  Objects are logged using the toString()
		 * method.
		 * **/
		public function warn(message:Object):void{
			log(WARN, message);
		}
		
		/**
		 * Log a message under the level of ERROR.  message may be of type String,
		 * Error, or Object.  Strings are used directly.  Error's messages
		 * and stackTraces are logged.  Objects are logged using the toString()
		 * method.
		 * **/
		public function error(message:Object):void{
			log(ERROR, message);
		}
		
		/**
		 * Log a message under the level of FATAL.  message may be of type String,
		 * Error, or Object.  Strings are used directly.  Error's messages
		 * and stackTraces are logged.  Objects are logged using the toString()
		 * method.
		 * **/
		public function fatal(message:Object):void{
			log(FATAL, message);
		}
		
		//********************************************
		// Helpers
		//********************************************

		/**
		 * Helper method that will (if extendedLogging == true) attempt
		 * to retrieve location information by creating an Error and parsing
		 * the stack trace.  This appears to be the only method available.
		 * **/
		protected function parseLocation(entry:LoggingEntry):void{
			try {
				var err:Error = new Error();
				var stack:Array = err.getStackTrace().split('\n', 10);
				
				var callingLine:String = stack[3];
				if (callingLine.indexOf("\tat com.edgeti.EdgeUtils.logger::Logger/") == 0){
					callingLine = stack[4];
				}
				
				var start:int;
				var stop:int;
				
				//Class Name
				start = callingLine.indexOf("at ", 0) + 3;
				stop = callingLine.indexOf("/", start);
				entry.className = callingLine.substr(start, stop-start);
				
				//Method name
				start = stop + 1;
				stop = callingLine.indexOf("[", start);
				entry.methodName = callingLine.substr(start, stop-start);
				
				//File name
				start = callingLine.indexOf("\\src", stop);
				stop = callingLine.indexOf(":", start);
				entry.fileName = callingLine.substr(start, stop-start);
				
				//Line Number
				start = stop + 1;
				stop = callingLine.indexOf("]", start);
				entry.lineNumber = callingLine.substr(start, stop-start);
			} catch (e:Error){
				//If not running in debug version of flash player,
				//err.getStackTrace() is null and throws an error
				//So catch it here and move on smoothly
				entry.className = "Not available in non-debug mode.";
			}
			
		}
	}
}