package com.edgeti.EdgeUtils.layout.managers
{
	import com.edgeti.EdgeUtils.DynamicPanel.DynamicPanel;
	import com.edgeti.EdgeUtils.DynamicPanel.PanelEvent;
	
	import mx.containers.Canvas;
	import mx.core.UIComponent;
	import mx.effects.Move;
	import mx.effects.Parallel;
	import mx.effects.Resize;
	import mx.effects.Sequence;
	import mx.events.EffectEvent;
	import mx.events.ResizeEvent;

	/**
	 * A completely free-form layourt manager that allows the placement
	 * of windows anywhere on the screen.  Minimized windows are not
	 * moved automatically and maximized windows are simply sized to the
	 * maximum dimensions.
	 * 
	 * **/
	public class AbsoluteLayoutManager extends LayoutManagerBase
	{
		//Buffer used for maximize
		protected static const BUFFER:int = 5;
		
		public function AbsoluteLayoutManager(view:Canvas)
		{
			super(view);
			
			name = "Absolute";
		}
		
		
		//************************************
		// Apply / Reset Layout
		//************************************
		
		override public function applyLayout(animate:Boolean):void{
			var panel:DynamicPanel;
			var i:int;
			
			var maximizedPanels:Array = new Array();
			
			//Split panels by their state
			for (i = 0; i < items.length; ++i){
				panel = items.getItemAt(i) as DynamicPanel;
				switch (panel.windowState){
					case DynamicPanel.WINDOW_STATE_MAXIMIZED:
						maximizedPanels.push(panel);
						break;
				}
			}
			
			//This saves a lot of duplicate code
			var duration:Number = animate ? 1000 : 0;
			
			applyLayoutHelper(maximizedPanels, duration);
		}
		
		protected function applyLayoutHelper(maxiPanels:Array, duration:Number):void{
			var panel:DynamicPanel;
			var xTo:Number;
			var yTo:Number;
			var wTo:Number;
			var hTo:Number;
			var i:int;
			var move:Move;
			var resize:Resize;
			
			var animations:Array = new Array();
			//Panels in maximized state
			for (i = 0; i < maxiPanels.length; ++i){
				panel = maxiPanels[i] as DynamicPanel;
				xTo = BUFFER;
				yTo = BUFFER;
				move = createMoveEffectInstance(panel, xTo, yTo);
				//move == null if no move is required
				if (move){
					animations.push(move);
				}
				wTo = getViewWidth() - 2 * BUFFER;
				hTo = getViewHeight() - 2 * BUFFER;
				resize = createResizeEffectInstance(panel, wTo, hTo);
				//resize == null if no resize is required
				if (resize){
					animations.push(resize);
				}
			}
			
			// Only animate if there are items to animate.
			if (animations.length > 0)
			{
				_parallel = new Parallel();
				_parallel.children = animations;
				_parallel.duration = duration;
				_parallel.play();
				_parallel.addEventListener(EffectEvent.EFFECT_END, onEffectEnd);
			}
		}
		
		//************************************
		// Managed Items Changes
		//************************************
		
		override protected function addPanelListeners(panel:DynamicPanel):void{
			super.addPanelListeners(panel);
			
			panel.allowClose = true;
			panel.allowResize = panel.windowState == DynamicPanel.WINDOW_STATE_DEFAULT;
			panel.allowMinimize = true;
			panel.allowMaximize = true;
			panel.allowMove = panel.windowState != DynamicPanel.WINDOW_STATE_MAXIMIZED;
		}
		
		override protected function onItemAdd(items:Array, location:int):void{
			super.onItemAdd(items, location);
			var xTo:Number;
			var yTo:Number;
			var wTo:Number;
			var hTo:Number;
			
			//Pretty safe assumption only one panel added at atime
			var panel:DynamicPanel = items[0] as DynamicPanel;
			switch (panel.windowState){
				case DynamicPanel.WINDOW_STATE_MAXIMIZED:
					xTo = BUFFER;
					yTo = BUFFER;
					wTo = getViewWidth() - 2 * BUFFER;
					hTo = getViewHeight() - 2 * BUFFER;
					
					panel.move(xTo, yTo);
					panel.width = wTo;
					panel.height = hTo;
					break;
				case DynamicPanel.WINDOW_STATE_MINIMIZED:
					panel.width = DynamicPanel.MIN_WIDTH;
					panel.height = DynamicPanel.MIN_HEIGHT;
					break;
				default:
					//By default, do nothing
					break;
			}
		}
		
		override protected function onItemReset():void{
			super.onItemReset();
			applyLayout(false);
		}
		
		//************************************
		// Minimize / Maximize
		//************************************
		
		override protected function onPanelMinimize(evt:PanelEvent):void{
			var panel:DynamicPanel = evt.target as DynamicPanel;
			
			movePanel(panel, panel.x, panel.y, DynamicPanel.MIN_WIDTH, DynamicPanel.MIN_HEIGHT, false);
		}
		
		override protected function onPanelMaximize(evt:PanelEvent):void{
			var panel:DynamicPanel = evt.target as DynamicPanel;
			//Move
			var xTo:Number = BUFFER;
			var yTo:Number = BUFFER;
			//Resize
			var wTo:Number = getViewWidth() - 2 * BUFFER;
			var hTo:Number = getViewHeight() - 2 * BUFFER;
			
			movePanel(panel, xTo, yTo, wTo, hTo, true);
		}
		
		override protected function onPanelRestore(evt:PanelEvent):void{
			var panel:DynamicPanel = evt.target as DynamicPanel;
			panel.allowResize = true;
			panel.allowMove = true;
			
			var xTo:Number = panel.x;
			var yTo:Number = panel.y;
			if (evt.previousState == DynamicPanel.WINDOW_STATE_MAXIMIZED){
				xTo = evt.oldX;
				yTo = evt.oldY;
			}
			var moveFirst:Boolean = evt.previousState == DynamicPanel.WINDOW_STATE_MINIMIZED;
			
			movePanel(panel, xTo, yTo, evt.oldWidth, evt.oldHeight, moveFirst);
		}
		
		protected function movePanel(panel:UIComponent, xTo:Number, yTo:Number,
										wTo:Number, hTo:Number, moveFirst:Boolean):void{
			
			var move:Move = createMoveEffectInstance(panel, xTo, yTo);
			var resize:Resize = createResizeEffectInstance(panel, wTo, hTo);
			
			var animations:Array = new Array();
			if (resize){
				animations.push(resize);
			}
			if (move){
				animations.push(move);
			}
			
			if (moveFirst){
				animations.reverse();
			}
			
			var sequence:Sequence = new Sequence();
			sequence.children = animations;
			sequence.duration = 400;
			sequence.play(); 
		}
		
		//************************************
		// Container Resize
		//************************************
		
		override protected function onContainerResize(evt:ResizeEvent):void{
			super.onContainerResize(evt);
			
			for each (var panel:DynamicPanel in _items){
				if (panel.windowState == DynamicPanel.WINDOW_STATE_MAXIMIZED){
					panel.width = getViewWidth() - 2 * BUFFER;
					panel.height = getViewHeight() - 2 * BUFFER;
				}
			}
		}
		
		//************************************
		// Focus Item
		//************************************
		
		override public function focusPanel(panel:UIComponent):void{
			_view.setChildIndex(panel, _view.numChildren - 1);
		}
	}
}