package com.edgeti.EdgeUtils.layout
{
	import com.edgeti.EdgeUtils.DynamicPanel.DynamicPanel;
	import com.edgeti.EdgeUtils.ImageCanvas;
	import com.edgeti.EdgeUtils.layout.managers.*;
	
	import flash.display.DisplayObject;
	
	import mx.collections.ArrayCollection;
	import mx.core.UIComponent;

	/**
	 * A managed canvas employs one LayoutManager at a time to handle DynamicPanel
	 * positioning, sizing, minimizing and maximizing.  It is possbile to flip between
	 * various LayoutManagers at run-time without losing panels.
	 * 
	 * The following LayoutManagers are currently available:
	 * 
	 * CascadeLayoutManager (0)
	 * VerticalScrollLayoutManager (1)
	 * BuilderLayoutManager (2)
	 * AbsoluteLayoutManager(3)
	 * 
	 * Changing LayoutManagers is done by calling:
	 * 
	 * setLayoutManagerIndex(index:int);
	 * 
	 * Panels are added using the addManagedChild method and
	 * removed with the removeChild method.
	 * **/
	public class ManagedCanvas extends ImageCanvas
	{
		private var _selectedLayoutManager:LayoutManagerBase;
		private var _cascadeManager:CascadeLayoutManager;
		private var _vertManager:VerticalScrollLayoutManager;
		private var _buildManager:BuilderLayoutManager;
		private var _absoluteManager:AbsoluteLayoutManager;
		
		[Bindable]
		protected var _managedChildren:ArrayCollection;
		[Bindable]
		public var managers:ArrayCollection;
		
		//************************************
		// Init
		//************************************
		
		public function ManagedCanvas()
		{
			super();
			
			_managedChildren = new ArrayCollection();
			
			initManagers();
		}
		
		protected function initManagers():void{
			managers = new ArrayCollection();
			
			_cascadeManager = new CascadeLayoutManager(this);
			managers.addItem(_cascadeManager);
			
			_vertManager = new VerticalScrollLayoutManager(this);
			managers.addItem(_vertManager);
			
			_buildManager = new BuilderLayoutManager(this);
			managers.addItem(_buildManager);
			
			_absoluteManager = new AbsoluteLayoutManager(this);
			managers.addItem(_absoluteManager);
		}
		
		//************************************
		// Managed Children
		//************************************
		
		[Bindable]
		public function get managedChildren():ArrayCollection{
			return _managedChildren;
		}
		
		public function set managedChildren(val:ArrayCollection):void{
			
		}
		
		//************************************
		// Add / Remove / Set
		//************************************
		
		public function addManagedChild(child:DynamicPanel):void{
			_managedChildren.addItem(child);
			child.isManaged = true;
			
			addChild(child);
		}
		
		override public function setChildIndex(child:DisplayObject, index:int):void{
			//TODO: Do we care about changing z order?
			
			super.setChildIndex(child, index);
		}
		
		override public function removeChild(child:DisplayObject):DisplayObject{
			var index:int = _managedChildren.getItemIndex(child);
			if (index != -1){
				_managedChildren.removeItemAt(index);
			}
			
			return super.removeChild(child);
		}
		
		//************************************
		// Public Methods
		//************************************
		
		public function closeAll():void{
			var panel:DynamicPanel;
			while (_managedChildren.length > 0){
				panel = _managedChildren.getItemAt(0) as DynamicPanel;
				panel.closePanel();
			}
		}
		
		public function minimizeAll():void{
			for each (var panel:DynamicPanel in _managedChildren){
				if (panel.windowState != DynamicPanel.WINDOW_STATE_MINIMIZED){
					panel.minimizePanel();
				}
			}
		}
		
		public function maximizeAll():void{
			for each (var panel:DynamicPanel in _managedChildren){
				if (panel.windowState != DynamicPanel.WINDOW_STATE_MAXIMIZED){
					panel.maximizePanel();
				}
			}
		}
		
		public function focusPanel(panel:UIComponent):void{
			_selectedLayoutManager.focusPanel(panel);
		}
		
		//************************************
		// Layout Chooser
		//************************************
		
		public function setLayoutManager(manager:LayoutManagerBase):void{
			if (_selectedLayoutManager != null && _selectedLayoutManager != manager)
			{
				if (_selectedLayoutManager.transitioning)
				{
					// Cancel the effect if it is playing so we get smooth transitions.
					_selectedLayoutManager.stopTransition();
				}
				
				_selectedLayoutManager.items = new ArrayCollection();
			}
			
			_selectedLayoutManager = manager;
			_selectedLayoutManager.items = _managedChildren;
			
			updateView();
		}
		
		public function setLayoutManagerIndex(index:int):void{
			if (index < managers.length){
				setLayoutManager(managers.getItemAt(index) as LayoutManagerBase);
			}
		}
		
		public function getLayoutManagerIndex():int{
			if (_selectedLayoutManager){
				for (var i:int = 0; i < managers.length; ++i){
					if (managers.getItemAt(i) == _selectedLayoutManager){
						return i;
					}
				}
			}
			
			return 0;
		}
			
		private function updateView():void
		{
			if (_managedChildren != null && _selectedLayoutManager != null)
			{
				_selectedLayoutManager.applyLayout(true);
			}
		}
		
		public function refreshView(animiate:Boolean=true):void{
			if (_managedChildren != null && _selectedLayoutManager != null)
			{
				_selectedLayoutManager.applyLayout(animiate);
			}
		}
	}
}