package com.edgeti.EdgeUtils
{
	/**
	 * Class to store colors as seperate RGB values, either in uint or Number form
	 **/
	public class RgbColor
	{
		private var _hexColor:uint;
		private var _nRed:Number;
		private var _nGreen:Number;
		private var _nBlue:Number;
		private var _r:uint;
		private var _g:uint;
		private var _b:uint;
		
		/**
		 * Ctor: default is to read in uint values
		 **/
		public function RgbColor(red:uint = 0, green:uint = 0, blue:uint = 0)
		{
			_r = red;
			_g = Math.max(green, 0x10); // avoid 0x000F00 color bug in degrafa - green values from 0x000100 - 0x000F00 will draw red
			_b = blue;
			
			var scalar:Number = 1.0/255.0; // save some CPU by dividing once
			_nRed = red * scalar;
			_nGreen = green * scalar;
			_nBlue = blue * scalar;
			
			_hexColor = blue + green*0x100 + red*0x10000;
		}
		
		/**
		 * Read in values as Numbers between 0 and 1
		 **/
		public function setAsNumber(red:Number, green:Number, blue:Number):void{
			red = Math.min(red, 1.0);
			red = Math.max(red, 0);
			green = Math.min(green, 1.0);
			green = Math.max(green, 0);
			blue = Math.min(blue, 1.0);
			blue = Math.max(blue, 0);
			
			_nRed = red * 255;
			_nGreen = green * 255;
			_nBlue = blue * 255;
			
			_r = (uint)(_nRed);
			_r = Math.min(_r, 255);
			_r = Math.max(_r, 0);
			_g = (uint)(_nGreen);
			_g = Math.min(_g, 255);
			_g = Math.max(_g, 0x10); // avoid 0x000F00 color bug in degrafa - green values from 0x000100 - 0x000F00 will draw red
			_b = (uint)(_nBlue);
			_b = Math.min(_b, 255);
			_b = Math.max(_b, 0);
			
			_hexColor = _b + _g*0x100 + _r*0x10000;
		}
		
		/**
		 * Copy 'constructor'
		 **/
		 public function setAsRgbColor(color:RgbColor):void{
		 	this(color.red, color.green, color.blue);
		 }

		
		//------------------- Operations
		public function subtract(color:RgbColor):RgbColor{
			return new RgbColor(_r - color.red, _g - color.green, _b - color.blue);
		}
		
		public function add(color:RgbColor):RgbColor{
			return new RgbColor(_r + color.red, _g + color.green, _b + color.blue);
		}
		
		public function mult(scalar:Number):RgbColor{
			var result:RgbColor = new RgbColor(_r, _g, _b);			
			result.setAsNumber(result.numRed*scalar, result.numGreen*scalar, result.numBlue*scalar);
			return result;
		}
		
		//------------------- Getters
		
		/**
		 * Get the color as a combined uint in the 0xRRGGBB format
		 **/
		public function get hexColor():uint{			
			return _hexColor;
		}
		
		public function get red():uint{
			return _r;
		}
		
		public function get green():uint{
			return _g;
		}
		
		public function get blue():uint{
			return _b;
		}
		
		public function get numRed():Number{
			return _nRed;
		}
		
		public function get numGreen():Number{
			return _nGreen;
		}
		
		public function get numBlue():Number{
			return _nBlue;
		}
		
		//--------------- Misc
		public function toString():String{
			return "("+Math.round(_nRed)+", "+Math.round(_nGreen)+", "+Math.round(_nBlue)+") ";
		}
	}
}